/** @file   redrawqueue.cpp
 * @brief   Implementation of RedrawQueue - class.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#include "redrawqueue.h"
#include "utils.h"
using std::multimap;
using std::pair;


namespace WeWantWar {

/** Constructor
 */
RedrawQueue::RedrawQueue()
{

}



/** Destructor
 */
RedrawQueue::~RedrawQueue()
{
  this->clear();
}


/** Adds new item to this queue
 */
void RedrawQueue::add( int pri, int x, int y, SpType sptype, void* pGfx )
{
  QueueItem* pQ = new QueueItem;
  pQ->drawX = x;
  pQ->drawY = y;
  pQ->type = sptype;
  pQ->blendmode = BMODE_NORMAL;
  pQ->alpha = 0;
  pQ->pGfx = pGfx;
  pQ->rot = 0;
  
  pair<int, QueueItem*> p( pri, pQ );
  m_bitmapQueue.insert( p );
}



/** Adds new item to redraw queues
 */
void RedrawQueue::add( int pri, int x, int y, SpType sptype,
                       Blendmode blend, int alpha, void* pGfx )
{
  QueueItem* pQ = new QueueItem;
  pQ->drawX = x;
  pQ->drawY = y;
  pQ->type = sptype;
  pQ->blendmode = blend;
  pQ->alpha = alpha;
  pQ->pGfx = pGfx;
  pQ->rot = 0;
  
  pair<int, QueueItem*> p( pri, pQ );
  m_bitmapQueue.insert( p );
}



/** Adds a rotated sprite
 */
void RedrawQueue::addRotatedSprite( int pri, int x, int y, fixed rot, void* pGfx )
{
  QueueItem* pQ = new QueueItem;
  pQ->drawX = x;
  pQ->drawY = y;
  pQ->type = SPTYPE_SPRITE;
  pQ->blendmode = BMODE_NORMAL;
  pQ->alpha = 0;
  pQ->pGfx = pGfx;
  pQ->rot = rot;
  
  pair<int, QueueItem*> p( pri, pQ );
  m_bitmapQueue.insert( p );
}




/** Adds new triangle
 */
void RedrawQueue::addTriangle( int pri, int polytype, BITMAP* pTex,
                               V3D_f* pV1, V3D_f* pV2, V3D_f* pV3 )
{
  
  PolyItem* pP = new PolyItem;
  pP->polytype = polytype;
  pP->vtxcount = 3;
  pP->pTex = pTex;
  pP->vertices[0] = *pV1;
  pP->vertices[1] = *pV2;
  pP->vertices[2] = *pV3;
  
  pair<int, PolyItem*> p( pri, pP );
  m_polyQueue.insert( p );
}



/** Adds new quad
 */
void RedrawQueue::addQuad( int pri, int polytype, BITMAP* pTex,
                           V3D_f* pV1, V3D_f* pV2, V3D_f* pV3, V3D_f* pV4 )
{

  PolyItem* pP = new PolyItem;
  pP->polytype = polytype;
  pP->vtxcount = 4;
  pP->pTex = pTex;
  pP->vertices[0] = *pV1;
  pP->vertices[1] = *pV2;
  pP->vertices[2] = *pV3;
  pP->vertices[3] = *pV4;
  
  pair<int, PolyItem*> p( pri, pP );
  m_polyQueue.insert( p );
}




/** Draws the contents from the queue
 */
void RedrawQueue::redraw( int pri, BITMAP* pTarget )
{
  pair< BitmapIterator, BitmapIterator > begEnd =
    m_bitmapQueue.equal_range( pri );
    
  for ( BitmapIterator it = begEnd.first; it != begEnd.second; it++ ) {
    QueueItem* pQ = it->second;
    switch ( pQ->type ) {
      case ( SPTYPE_BITMAP ): {
        // Normal direct blitting.
        BITMAP* pB = reinterpret_cast<BITMAP*>( pQ->pGfx );
        blit( pB, pTarget, 0,0, pQ->drawX, pQ->drawY, pB->w, pB->h );
        break;
      }
      case ( SPTYPE_SPRITE ): {
        // This should be drawn as sprite
        BITMAP* pB = reinterpret_cast<BITMAP*>( pQ->pGfx );
        if ( pQ->rot == 0 ) {
          // We draw non-rotated sprite.
          if ( pQ->blendmode == BMODE_NORMAL ) {
            draw_sprite( pTarget, pB, pQ->drawX, pQ->drawY );
          } else {
            this->setBlenderFunction( pQ->blendmode, pQ->alpha );
            draw_trans_sprite( pTarget, pB, pQ->drawX, pQ->drawY );
          }
          
        } else {
          // Rotated sprites do not support any blending, so we just
          // draw it.
          rotate_sprite( pTarget, pB, pQ->drawX, pQ->drawY, pQ->rot );
          
        }
        break;
      }
      case ( SPTYPE_RLE ): {
        // The RLE-sprites cannot be rotated so we just check the blending
        // mode and draw the sprite then.
        RLE_SPRITE* pS = reinterpret_cast<RLE_SPRITE*>( pQ->pGfx );
        if ( pQ->blendmode == BMODE_NORMAL ) {
          draw_rle_sprite( pTarget, pS, pQ->drawX, pQ->drawY );
        } else {
          this->setBlenderFunction( pQ->blendmode, pQ->alpha );
          draw_trans_rle_sprite( pTarget, pS, pQ->drawX, pQ->drawY );
        }
        break;
      }
      default: {
        break;
      }
    }
  }
  
  this->drawPolygons( pri, pTarget );
}






/** Draws the polygons
 */
void RedrawQueue::drawPolygons( int pri, BITMAP* pTarget )
{
  pair< PolyIterator, PolyIterator > begEnd =
    m_polyQueue.equal_range( pri );

  for ( PolyIterator it = begEnd.first; it != begEnd.second; it++ ) {
    PolyItem* pQ = it->second;
    if ( pQ->vtxcount == 3 ) {
      triangle3d_f( pTarget, pQ->polytype, pQ->pTex,
                    &pQ->vertices[0], &pQ->vertices[1], &pQ->vertices[2] );
    } else {
      quad3d_f( pTarget, pQ->polytype, pQ->pTex,
                &pQ->vertices[0], &pQ->vertices[1],
                &pQ->vertices[2], &pQ->vertices[3] );
    }
  }
}



/** Clears the queues
 */
void RedrawQueue::clear()
{
  BitmapIterator it = m_bitmapQueue.begin();
  while ( it != m_bitmapQueue.end() ) {
    delete it->second;
    it++;
  }
  m_bitmapQueue.clear();
  
  PolyIterator it2 = m_polyQueue.begin();
  while ( it2 != m_polyQueue.end() ) {
    delete it2->second;
    it2++;
  }
  m_polyQueue.clear();
}


/** Sets the blender function
 */
void RedrawQueue::setBlenderFunction( Blendmode blend, int alpha )
{
  switch ( blend ) {
    case ( BMODE_ADD ): {
      set_add_blender( 0,0,0, alpha );
      break;
    }
    case ( BMODE_TRANS ): {
      set_trans_blender( 0,0,0, alpha );
      break;
    }
    case ( BMODE_MULTIPLY ): {
      set_multiply_blender( 0,0,0, alpha );
      break;
    }
    case ( BMODE_BURN ): {
      set_burn_blender( 0,0,0, alpha );
      break;
    }
    case ( BMODE_DODGE ): {
      set_dodge_blender( 0,0,0, alpha );
      break;
    }
    case ( BMODE_PREDATOR ): {
      set_blender_mode( Utils::predator_blend_b16,
                        Utils::predator_blend_b16,
                        Utils::predator_blend_b16, 0,0,0, alpha );
      break;
    }
  }
}


} // end of namespace
